// 默认的重试配置
const defaultRetryOptions = {
    retries: 5, // 最大重试次数
    retryDelay: (retryCount) => Math.min(1000 * 2 ** retryCount, 3000), // 指数退避，最多等待 3 秒
    shouldRetry: (error) => {
        // 判断是否可以重试的条件
        if (!error.response) {
            // 网络错误（没有响应对象）
            return true;
        }
        console.log("error", error.response);
        
        const { status } = error.response;
        // 对于 429（请求过多）和 5xx（服务器错误）进行重试
        return status === 429 || (status >= 500 && status < 600);
    },
};

// Axios 拦截器：实现重试逻辑
axios.interceptors.response.use(
    (response) => response, // 请求成功时直接返回响应
    async (error) => {
        const config = error.config;
        if (!config) {
            return Promise.reject(error);
        }

        // 初始化重试计数器
        config.__retryCount = config.__retryCount || 0;

        // 获取当前的重试配置
        const { retries, retryDelay, shouldRetry } = {
            ...defaultRetryOptions,
            ...config.retryOptions,
        };

        // 判断是否可以重试
        if (config.__retryCount < retries && shouldRetry(error)) {
            config.__retryCount += 1; // 增加重试计数

            const delay = retryDelay(config.__retryCount); // 获取当前重试延迟时间
            console.warn(`Retrying request (${config.__retryCount}/${retries}) after ${delay}ms...`);

            // 等待一段时间后再进行重试
            await new Promise((resolve) => setTimeout(resolve, delay));

            // 重试请求
            return axios(config);
        }

        // 如果超过重试次数，拒绝请求并返回错误
        return Promise.reject(error);
    }
);